//================================================================================================================================================\\
//================================================================================================================================================\\
//================================================================================================================================================\\
//                                                                                                                                                \\
//                                                                  Map Engine 2                                                                  \\
//                                                                                                                                                \\
//================================================================================================================================================\\
//================================================================================================================================================\\
//================================================================================================================================================\\

// TILES
// GetNumTiles() - To get number of tiles in the tileset
// GetTileImage(tile_index) - To get the image for tile_index
// GetTileWidth() - To get the tilesets width
// GetTileHeight() - To get the tilesets height
// GetNextAnimatedTile(tile) - Gets the next tile in the animation sequence of 'tile' note that if the return value is 'tile' the tile is not animated.
// GetTileDelay(tile) - Gets the animation delay of the tile if it returns 0, the tile is not animated

// MAP
// GetCurrentMap() - To get the maps name
// GetTile(x, y, layer) - To get a tile via reference
// GetTileName(tile_index) - To get the tiles index, then we will know what image to show for that tile

// LAYERS
// GetNumLayers() - To get the max number of layers
// GetLayerName(layer_index) - To get the layers name
// GetLayerWidth(layer_index) - To get the layers width in amount of tiles
// GetLayerHeight(layer_index) - To get the layers height in amount of tiles
// IsLayerVisible(layer_index) - To find out if this layer is being rendered

// PEOPLE
// GetPersonList() - To get a list of the names of all the people on the map
// GetPersonX() - To get the persons x co-ord
// GetPersonY() - To get the persons y co-ord
// GetPersonLayer() - To get the persons layer
// GetPersonDirection() - To get the persons current direction
// GetPersonSpriteset() - To get the persons spriteset object
// GetPersonBase() - To get the persons base object

// SPRITE
// sprite_object.images - To get array of the images in the sprite
// sprite_object.directions - To get array of the directions in the sprite
// directions_object.frames - To get array of frames in the direction

var mapEngine = new NewMapEngine();

function NewMapEngine()
{
  this.mapName = undefined; // Map name
  this.mapHeight = undefined; // Map height in tiles
  this.mapWidth = undefined; // Map width in tiles
  this.tileHeight = undefined; // Tile height
  this.tileWidth = undefined; // Tile width
  
  this.layers = undefined; // Array for each layer
  this.tileSet = undefined; // Array for the tileset
  this.zones = undefined; // Array for the zones
  this.person = undefined; // Array for the people
  this.tilesToShow = undefined; // Array for tiles within the camera's scope
  
  this.x = undefined; // Top left tile's x
  this.y = undefined; // Top left tile's y
  this.w = undefined; // Amount of tiles screen can hold
  this.h = undefined; // Amount of tiles screen can hold
  this.tileW = undefined; // Right tile's x
  this.tileH = undefined; // Bottom tile's y
  
  this.tempTile = undefined; // Used to refer to during tile drawing;
  
  this.infoCollected = false; // Used to make sure we only grab the map info and tiles once
  
  this.layerRenderQueue = undefined; // Queue of renderings for each layer, and the chance to render before and after each layer 
  
  this.personObject = function(person)
  {
    this.name = person; // Set person's name
    this.x = GetPersonX(person); // Set person's x
    this.y = GetPersonY(person); // Set person's y
    this.h = GetPersonValue(person, "height"); // Set person's height
    this.w = GetPersonValue(person, "width"); // Set person's width
    this.layer = GetPersonLayer(person); // Set person's layer
    this.base = GetPersonBase(person); // Set person's base
    
    var sprite = GetPersonSpriteset(person); // Get person's sprite
    var directions = sprite.directions; // Get each direction in the sprite
    
    for (var d = 0; d < directions.length; d++) // For each direction
    {
      this[directions[d].name] = []; // Add the direction as a property in person, and make it an array
      var frames = directions[d].frames; // Get amount of frames in this direction
      for (var f = 0; f < frames.length; f++) // For each frame
      {
        this[directions[d].name][this[directions[d].name].length] = frames[f]; // Store the frames index into the array we just made
      }
    }
    this.images = sprite.images; // Store each image in the spriteset in an array, then we can call to them via index
  }
}

NewMapEngine.prototype.getMapInfo = function()
{
  this.getTileSet(); // Gets the tileset
  this.getMap(); // Gets the maps info
  this.getPeople(); // Gets every person on the map
  this.getRenderQueue(); // Gets render queue for each layer
  this.infoCollected = true; // Stop engine from trying to gather this information again
}

NewMapEngine.prototype.getTileSet = function()
{
  this.tileHeight = GetTileHeight(); // Get the tile height
  this.tileWidth = GetTileWidth(); // Get the tile width
  
  this.tileSet = []; // Make a new array for the tileset
  for (var i = 0; i < GetNumTiles(); i++) // For every tile in the tileset
  {
    var tile = new Object;
    tile.image = GetTileImage(i); // We take the tiles image
    tile.animated = false; // Is tile animated - set to false for default
    tile.index = i; // Tile index in animation - set to self for default
    if (GetNextAnimatedTile(i) != i) { tile.animated = true; } // If tile is animated change value to true
    tile.nextTile = GetNextAnimatedTile(i); // Gets the next tile in the animation
    tile.delay = GetTileDelay(i) * 16.6; // Gets the delay between the animation, 16.6 is for each frame
    tile.time = GetTime(); // Get current time
    this.tileSet[this.tileSet.length] = tile; // And save it
  }
}

NewMapEngine.prototype.getPersonList = function()
{
  var array = []; // Create a new array that will store the peoples names
  for (var p in this.person) // Do once for each person we have stored
  {
    array[array.length] = p; // Add this person's name to our new array
  }
  return array; // Return our array of names
}

NewMapEngine.prototype.getPeople = function()
{
  this.person = []; // Create a new array for people on the map
  var list = GetPersonList(); // Get a list of all people on the map's names
  for (var p = 0; p < list.length; p++) // Do once for each person on the map
  {
    this.person[this.person.length] = new this.personObject(list[p]); // Store in our person array
  }
}

NewMapEngine.prototype.getMap = function()
{
  this.mapName = GetCurrentMap(); // Get the maps name
  this.mapHeight = GetLayerHeight(0); // Get the maps height in tiles
  this.mapWidth = GetLayerWidth(0); // Get the maps width in tiles
  
  this.h = Math.floor(camera.h / this.tileHeight) + 2; // Gets amount of tiles maps height can hold (+2 to allow for half tiles on either side etc)
  this.w = Math.floor(camera.w / this.tileWidth) + 2; // Gets amount of tiles maps width can hold (+2 to allow for half tiles on either side etc)
  
  this.layers = []; // Make a new array for the layers
  
  for (var i = 0; i < GetNumLayers(); i++) // For every layer on this map
  {
    var layer = new Object;
    layer.name = GetLayerName(i); // Get the layers name
    layer.visible = IsLayerVisible(i); // Is the layer invisible
    layer.y = []; // Make a new array for y axis
    this.layers[this.layers.length] = layer; // And save it
  }
  
  for (var l = 0; l < this.layers.length; l++) // For every layer on this map
  {
    for (var y = 0; y < this.mapHeight; y++) // For every tile the map has in height
    {
      this.layers[l].y[y] = []; // Make a new array for x axis
      for (var x = 0; x < this.mapWidth; x++) // For every tile the map has in width
      {
        var yX = new Object;
        yX.tile = GetTile(x, y, l); // Get this tiles index to the tileset
        // We can also store other things here - is trigger here?
        this.layers[l].y[y][x] = yX; // And save it
      }
    }
  }  
}

NewMapEngine.prototype.getRenderQueue = function()
{
  this.layerRenderQueue = []; // Create new array for each layer, for rendering, pre-rendering and post-rendering
  for (var i = 0; i < this.layers.length; i++) // Do once for each layer
  {
    var queue = new Object;
    queue.preRender = undefined; // Set pre-render function as string
    queue.postRender = undefined; // Set post-render function as string
    this.layerRenderQueue[this.layerRenderQueue.length] = queue; // Add to our array
  }
}

NewMapEngine.prototype.updatePerson = function()
{
  for (var i = 0; i < this.person.length; i++) // For each person in our person array
  {
    this.person[i].x = GetPersonX(this.person[i].name); // Update person's x
    this.person[i].y = GetPersonY(this.person[i].name); // Update person's y
  }
  this.person.sort(function (a,b) { return a.y-b.y; }); // Sort person array in order of y - for blitting reasons
}

NewMapEngine.prototype.updateAnimatedTiles = function()
{
  var currentTime = GetTime(); // Get current time - may change to frames
  for (var i = 0; i < this.tileSet.length; i++) // Do once for each tile in the tileset
  {
    if (this.tileSet[i].animated) // If tile is animated
    {
      if (currentTime > this.tileSet[i].time + this.tileSet[this.tileSet[i].index].delay) // If time is greater than the tiles last stored time plus the delay
      {
        this.tileSet[i].index = this.tileSet[this.tileSet[i].index].nextTile; // Set the index to the next in its animation
        this.tileSet[i].time = currentTime; // Set last stored time to now
      }
    }
  }
}

NewMapEngine.prototype.getPersonId = function(person)
{
  for (var i = 0; i < this.person.length; i++) // For each person in our person list
  {
    if (this.person[i].name == person) // If the person's name is the one we are looking for
    {
      return i; // Return person's id
    }
  }
  Abort("Person is not in this.person array"); // If not found
}

NewMapEngine.prototype.update = function()
{
  if (!this.infoCollected) 
  {
    this.getMapInfo(); 
  }
  // If this is the first run-through then get the map information, then set to collected
  else // If this is not the first run through then update the map state
  {
    this.updatePerson(); // Update each person's co-ords and sort the array in y-order
    this.updateAnimatedTiles(); // Update the animated tiles
    this.x = Math.floor(camera.x / this.tileWidth); // Set top left tile
    if (this.x + this.w > this.mapWidth) this.x = this.mapWidth - this.w; // If previous co-ord is off map then move onto maps edge
    this.y = Math.floor(camera.y / this.tileHeight); // Set top left tile
    if (this.y + this.h > this.mapHeight) this.y = this.mapHeight - this.h; // If previous co-ord is off map then move onto maps edge
    this.tileW = this.w + this.x; // Set how many tiles across to blit from this.x
    this.tileH = this.h + this.y; // Set how many tiles down to blit from this.y
  }
}

NewMapEngine.prototype.render = function()
{
  Rectangle(0, 0, 320, 240, CreateColor(0, 0, 0, 255));
  for (var i = 0; i < this.layerRenderQueue.length; i++) // Don once for every layer we are to render
  {
    // Pre-render will be run here
    if (this.layerRenderQueue[i].preRender) this.layerRenderQueue[i].preRender();
    this.layerRender(i);
    if (this.layerRenderQueue[i].postRender) this.layerRenderQueue[i].postRender();
    // Post-render will be run here
  }
  drawText(2, 37, "Camera is attached to " + camera.player); // FOR DEBUG - NOT NEEDED
  drawText(2, 47, "X = " + camera.x); // FOR DEBUG - NOT NEEDED
  drawText(2, 57, "Y = " + camera.y); // FOR DEBUG - NOT NEEDED
  drawText(2, 67, "W = " + camera.w); // FOR DEBUG - NOT NEEDED
  drawText(2, 77, "H = " + camera.h); // FOR DEBUG - NOT NEEDED
}


NewMapEngine.prototype.layerRender = function(l)
{
  for (var y = this.y; y < this.tileH; y++) // For each tile between the top and bottom of the screen
  {
    for (var x = this.x; x < this.tileW; x++) // For each tile between the left and the right of the screen
    {
      this.tempTile = this.tileSet[this.layers[l].y[y][x].tile].index; // Get the tiles index
      this.tileSet[this.tempTile].image.blit(camera.mapToScreenX(x * this.tileWidth), camera.mapToScreenY(y * this.tileHeight)); // Blit the tile at appropriate co-ords
    }
  }
  for (var p = 0; p < this.person.length; p++) // For each person in person array
  {
    if (this.person[p].layer == l) // If the person is on this layer
    {
      var x = camera.mapToScreenX(this.person[p].x - Math.ceil((this.person[p].base.x2 - this.person[p].base.x1) * 0.5) - this.person[p].base.x1 + 1); // Get person's x and work out where to blit person's image
      var y = camera.mapToScreenY(this.person[p].y - Math.ceil((this.person[p].base.y2 - this.person[p].base.y1) * 0.5) - this.person[p].base.y1 + 1); // Get person's y and work out where to blit person's image
      if (x + this.person[p].w > 0 && x < camera.w && y + this.person[p].h > 0 && y < camera.h) // If person's image is within the camera scope
      {
        Rectangle(x, y, this.person[p].w, this.person[p].h, CreateColor(0, 255, 0, 120));
        var index = this.person[p][GetPersonDirection(this.person[p].name)][GetPersonFrame(this.person[p].name)].index; // Get the person's frame/image index
        this.person[p].images[index].blit(x, y); // Blit that image to display the person
      }
    }
  }
}